const nextButton = document.getElementById('next-button');
const flashcardContainer = document.getElementById('flashcard-container');
const playAudioButton = document.getElementById('play-audio-button');
const audioPlayer = document.getElementById('audio-player');
const audioSource = document.getElementById('audio-source');
const showNoteButton = document.getElementById('show-note-button');
const reviseButton = document.getElementById('revise-button');
const showRevisedButton = document.getElementById('show-revised-button');
const downloadRevisedButton = document.getElementById('download-revised-button');
const noteContainer = document.querySelector('.note-container');
const noteContent = document.getElementById('note-content');

let flashcards = [];
let currentCardIndex = 0;
const revisedFlashcards = [];

function loadAndShuffleFlashcards() {
    fetch('flashcards.json')
        .then(response => response.json())
        .then(data => {
            flashcards = data;
            shuffleArray(flashcards);
            showFlashcard();
        })
        .catch(error => console.error('Error loading flashcards: ', error));
}

function showFlashcard() {
    if (flashcards.length === 0) {
        flashcardContainer.innerHTML = 'No flashcards available.';
        return;
    }

    const flashcard = flashcards[currentCardIndex];
    flashcardContainer.innerHTML = `
        <div class="flashcard" id="flashcard">
            <div class="card-front" id="card-front">
                <p>${flashcard.Question}</p>
            </div>
            <div class="card-back" id="card-back">
                <p>${flashcard.Answer}</p>
            </div>
        </div>
    `;

    audioSource.src = `audio/${flashcard.Answer}.mp3`;

    const flashCard = document.getElementById('flashcard');
    const cardFront = document.getElementById('card-front');
    const cardBack = document.getElementById('card-back');
    flashCard.addEventListener('click', () => {
        cardFront.style.transform = 'rotateY(180deg)';
        cardBack.style.transform = 'rotateY(0deg)';
    });

    flashCard.addEventListener('click', () => {
        cardFront.style.transform = 'rotateY(0deg)';
        cardBack.style.transform = 'rotateY(180deg)';
    });

    if (currentCardIndex === flashcards.length - 1) {
        nextButton.disabled = true;
        showRevisedButton.style.display = 'inline-block';
        downloadRevisedButton.style.display = 'inline-block';
    } else {
        nextButton.disabled = false;
        showRevisedButton.style.display = 'none';
        downloadRevisedButton.style.display = 'none';
    }

    hideNote();

    if (flashcard.Note) {
        showNoteButton.style.display = 'inline-block';
    } else {
        showNoteButton.style.display = 'none'; // Fix the syntax error here
    }
}

showNoteButton.addEventListener('click', () => {
    noteContainer.style.display = 'block';
    noteContent.textContent = flashcards[currentCardIndex].Note || '';
});

function hideNote() {
    noteContainer.style.display = 'none';
}

playAudioButton.addEventListener('click', () => {
    audioPlayer.load();
    audioPlayer.play();
    setTimeout(() => {
        audioPlayer.pause();
    }, 1000);
});

nextButton.addEventListener('click', function () {
    currentCardIndex = (currentCardIndex + 1) % flashcards.length;
    showFlashcard();
});

reviseButton.addEventListener('click', reviseFlashcard);
downloadRevisedButton.addEventListener('click', downloadRevisedFlashcards);

showRevisedButton.addEventListener('click', showRevisedFlashcards);

function shuffleArray(array) {
    for (let i = array.length - 1; i > 0; i--) {
        const j = Math.floor(Math.random() * (i + 1));
        [array[i], array[j]] = [array[j], array[i]];
    }
}

function reviseFlashcard() {
    const flashcardToRevise = flashcards[currentCardIndex];
    if (!isFlashcardInRevisedArray(flashcardToRevise)) {
        revisedFlashcards.push(flashcardToRevise);
    }
    showFlashcard();
}

function isFlashcardInRevisedArray(flashcard) {
    return revisedFlashcards.some(revised => {
        return (
            revised.Question === flashcard.Question &&
            revised.Answer === flashcard.Answer &&
            revised.Note === flashcard.Note
        );
    });
}

function showRevisedFlashcards() {
    flashcardContainer.innerHTML = '';
    revisedFlashcards.forEach((flashcard, index) => {
        flashcardContainer.innerHTML += `
            <div class="revised-flashcard">
                <p>Question: ${flashcard.Question}</p>
                <p>Answer: ${flashcard.Answer}</p>
                ${flashcard.Note ? `<p>Note: ${flashcard.Note}</p>` : ''}
            </div>
        `;
    });
}

function generateRevisedText() {
    const textContent = revisedFlashcards.map((flashcard, index) => {
        const noteText = flashcard.Note ? `Note: ${flashcard.Note}\n` : '';
        return `Question: ${flashcard.Question}\nAnswer: ${flashcard.Answer}\n${noteText}`;
    }).join('\n\n');
    return textContent;
}

function generateHash(content) {
    const hash = sha256.create();
    hash.update(content);
    return hash.hex().slice(0, 5); // Get the first 5 characters of the hash
}

function downloadRevisedFlashcards() {
    const textContent = generateRevisedText();
    const hash = generateHash(textContent); // Generate a hash from the content
    const fileName = `${hash}_flashcards.txt`; // Create the file name
    const blob = new Blob([textContent], { type: 'text/plain' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url;
    a.download = fileName; // Set the file name
    a.click();
    URL.revokeObjectURL(url);
}


loadAndShuffleFlashcards();
